import { expect } from 'chai';
import { describe } from 'mocha';
import { ethers } from 'hardhat';
import {
  FameContracts,
  deployContractsForTests,
  getSigners,
} from './helpers/deploy-helper';
import { Signers } from './types';
import { initializeContracts } from './helpers/contracts-initialization-helper';
import { BigNumber } from 'ethers';
import { TEST_CONTRACT_ADDRESSESS } from './helpers/expect-helpers';

import * as dotenv from 'dotenv';
const envFile = process.env.NODE_ENV ? `.env.${process.env.NODE_ENV}` : '.env';
dotenv.config({ path: envFile });

const contractAddresses = TEST_CONTRACT_ADDRESSESS;

describe('01 - All contracts set up and initialization tests', function () {
  let contracts: FameContracts;
  let signers: Signers;

  before(async function () {
    contracts = await deployContractsForTests({
      shouldResetNetwork: false,
      shouldSaveInfo: true,
    });
    signers = await getSigners();
    await initializeContracts({ contracts, signers });
    console.log(signers.admin.address);
  });

  describe('Deployment Checks', function () {
    it(`should deploy PaymentToken contract`, async function () {
      expect(contracts.paymentTokenContract.address).to.equal(
        contractAddresses.paymentTokenContract,
      );
    });

    it(`should deploy Escrow contract`, async function () {
      expect(contracts.escrowContract.address).to.equal(
        contractAddresses.escrowContract,
      );
    });

    it(`should deploy Offering contract`, async function () {
      expect(contracts.offeringTokenContract.address).to.equal(
        contractAddresses.offeringTokenContract,
      );
    });

    it('should deploy DataAccessPayAsYouGo contract', async function () {
      expect(contracts.dataAccessPayAsYouGoContract.address).to.equal(
        contractAddresses.dataAccessPayAsYouGoContract,
      );
    });

    it('should deploy DataAccessPayAsYouUse contract', async function () {
      expect(contracts.dataAccessPayAsYouUseContract.address).to.equal(
        contractAddresses.dataAccessPayAsYouUseContract,
      );
    });

    it('should deploy DataAccessSubscription contract', async function () {
      expect(contracts.dataAccessSubscriptionContract.address).to.equal(
        contractAddresses.dataAccessSubscriptionContract,
      );
    });

    it('should deploy Bourse contract', async function () {
      expect(contracts.bourseContract.address).to.equal(
        contractAddresses.bourseContract,
      );
    });

    it('should deploy Governance contract', async function () {
      expect(contracts.governanceContract.address).to.equal(
        contractAddresses.governanceContract,
      );
    });
  });

  describe('Ownership Transfer Checks', function () {
    it('should transfer PaymentToken ownership to Governance contract', async function () {
      const paymentTokenOwner = await contracts.paymentTokenContract.owner();
      expect(paymentTokenOwner).to.equal(contracts.governanceContract.address);
    });

    it('should transfer Bourse ownership to Governance contract', async function () {
      const bourseOwner = await contracts.bourseContract.owner();
      expect(bourseOwner).to.equal(contracts.governanceContract.address);
    });
  });

  describe('Escrow Contract Setup', function () {
    it('should set Bourse address in Escrow contract', async function () {
      const bourseAddressInEscrow = await contracts.escrowContract.getBourse();
      expect(bourseAddressInEscrow).to.equal(contracts.bourseContract.address);
    });
  });

  describe('Governance Contract Setup', function () {
    it('in Governance contract, Bourse contract address is set', async function () {
      const famebourseaddr = await contracts.governanceContract.getFameBourse();
      expect(famebourseaddr).to.equal(contracts.bourseContract.address);
    });
    it('should register FDE token in Governance contract and return array of registration index and address', async function () {
      const fdeCoinAddress: [string, BigNumber] =
        await contracts.governanceContract.getCoinAddress('FDE');
      expect(fdeCoinAddress[0]).to.equal(
        contracts.paymentTokenContract.address,
      );
      expect(fdeCoinAddress[1]).to.equal(0);
    });
    it('FDE coin is registered to the governance contract (noofCoins = 1)', async function () {
      // check how many coins are registered in governance contract
      // if number of coins (numcoins) is 0, it means FDE is not registered
      const numcoins = await contracts.governanceContract.getNoofCoins();
      expect(numcoins).to.equal(1);
    });
  });
});
